#region References

using System;
using System.Data;
using System.Diagnostics;
using System.Text;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.Common.Log;
using gov.va.med.vbecs.DAL.HL7AL;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;

#endregion

namespace gov.va.med.vbecs.DAL.HL7.Parsers
{
	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
	//	<Developer>Brian Tomlin</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>4/27/2004</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	//<summary>This class is responsible for handling incoming HL7 messages from the VBECS Patient Merge HL7 interface and updating the VBECS database with the merge data.</summary>

	#endregion 

	/// <summary>
	/// Class PatientMergeHL7Parser
	/// </summary>
	public class PatientMergeHL7Parser
	{
		#region Variables

		private const string INVALID_MERGE_FROM_PATIENT_NAME_INDICATOR = "MERGING INTO"; // CR 2931

		private const string LAST_UPDATE_USER_VALUE = "MPIF";
		private const string CARRIAGE_RETURN = "\x0D";
		private const string PATIENT_MERGE_MESSAGE_TYPE = "ADT~A40"; // CR 2975
		private const string ACK_MESSAGE_TYPE = "ACK";
		private const string ACK_TRIGGER_EVENT = "A40";

		private string _message;
		private string _fromPatientDfn;
		private string _fromPatientIcn;
		private string _toPatientDfn;
		private string _toPatientIcn;
		private string _toPatientSsn;
		private string _messageControlID;

		private string[] _msh;
		private string[] _pid;
		private string[] _mrg;

		private char[] _delimiters;

		private DataTable _dtPatientChange;

		private bool _fromPatientInVbecs = false;
		private bool _toPatientInVbecs = false;

        // Events Logger
        private readonly ILogger _eventsLogger =
            LogManager.Instance().LoggerLocator.GetLogger("SystemEvents");

		#endregion

		#region Public Methods

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5716"> 
		///		<ExpectedInput>Valid HL7 merge message containing 2 VBECS patients to be merged</ExpectedInput>
		///		<ExpectedOutput>Valid HL7ProtocolMessage with "AA" Ack code</ExpectedOutput>
		///	</Case>
		/// 
		///<Case type="1" testid ="5717"> 
		///		<ExpectedInput>Merge-from patient id not in VBECS; merge-to patient in VBECS</ExpectedInput>
		///		<ExpectedOutput>Valid HL7ProtocolMessage with "AA" Ack code</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5933"> 
		///		<ExpectedInput>Merge-from patient id in VBECS; merge-to patient not in VBECS</ExpectedInput>
		///		<ExpectedOutput>Valid HL7ProtocolMessage with "AA" Ack code</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="5934"> 
		///		<ExpectedInput>Null MessageString parameter</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="5937"> 
		///		<ExpectedInput>Null HL7Interface parameter</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5938"> 
		///		<ExpectedInput>Invalid HL7Interface parameter: CPRS used instead of Patient Merge</ExpectedInput>
		///		<ExpectedOutput>HL7Exception</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="6974"> 
		///		<ExpectedInput>Invalid HL7Interface parameter: invalid Message Type in message</ExpectedInput>
		///		<ExpectedOutput>HL7Exception</ExpectedOutput> 
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method parses the incoming patient merge HL7 message and determines if the two 
		/// patients included in the message are Blood Bank patients and can be merged.
		/// 
		/// CR 2423: VBECS will not return an error message to VistA even if an exception is 
		/// encountered while processesing the merge message or if one or both patients are 
		/// not in VBECS. 
		/// </summary>
		/// <param name="intParms"><see cref="HL7Interface"/> containing interface parameters for the 
		/// Patient Merge HL7 interface.</param>
		/// <param name="messageString">Valid HL7 ADT~A40 message.</param>
		/// <returns>HL7ProtocolMessage Acknowledgement message.</returns>
		public HL7ProtocolMessage ParseHL7Message(HL7Interface intParms, string messageString)
		{
			#region Check Parameters
			
			if( intParms == null )
			{
				throw( new ArgumentNullException( "intParms" ) );
			}
			//
			if( messageString == null || messageString == string.Empty )
			{
				throw( new ArgumentNullException( "messageString" ) );
			}
			//
			if( intParms.InterfaceName != InterfaceName.Patient_Merge.ToString() )
			{
				throw( new HL7Exception( "PatientMergeHL7Parser.ParseHL7Message can only be used with the Patient Merge HL7Interface" ) );
			}
			//
			// CR 2566
			if( HL7Utility.ParseGetHL7MessageTypeAndTriggerEventString( messageString ) != PATIENT_MERGE_MESSAGE_TYPE )
			{
				throw( new HL7Exception( "Invalid Message Type" ) );
			}

            

			#endregion

			try
			{
				// CR 2566
				_fromPatientInVbecs = false;
				_toPatientInVbecs = false;
				//
				// CR 2566
				this._message = messageString;
				//
				_dtPatientChange = HL7AL.PatientChange.GetPatientChangeDataTable();
				_dtPatientChange.Rows.Add( _dtPatientChange.NewRow() );
				//
				LoadDataFromHL7Message();
                //CR 3243
                //Check the value of processing id in the message
                if (this._msh[10] != String.Empty)
                {
                    char[] procId = this._msh[10].ToCharArray();
                    if (procId[0] != intParms.ProcessingId)
                    {
                        throw (new HL7Exception("Conflicting Processing ID in merge message, service is configured as " + intParms.ProcessingId + " and the merge message contains a processing ID of " + procId[0]));
                    }
                }
				//
				// Get Message Control ID for use in Message Log update.
				this._messageControlID = HL7Utility.GetMessageControlID( this._message );
				//
				// Check if VBECS patient exists
				if( ValidateVbecsPatientIdentity() )
				{
					HL7MessageLog.InsertMessageLog( HL7MessageLog.GetMessageDataForMessageLogInsert(messageString , MessageStatus.ProcessingIncomingMessage, intParms, Common.UpdateFunction.HL7PatientUpdateInterface) );
					//
					UpdatePatientChangeData();
					//
					// CR 2961
					HL7MessageLog.UpdateMessageStatus( MessageStatus.SuccessfullyCompleted, this._messageControlID, null, 0, intParms.InterfaceName, Common.UpdateFunction.HL7PatientMergeInterface );
				}
			}
			catch( Exception exc )
			{
				// CR 2423: rather than return an error to VistA, we write exception to event log and email interface administrator.
				//
				var messageBody = string.Concat("Error processing patient merge message: ", HL7Utility.GetInnerExceptionFromException(exc));
                //EventLogAppender
                _eventsLogger.Error("VBECS Merge Interface: " + messageBody);
				try
				{
					HL7MailMessage mailMessage = new HL7MailMessage();
					mailMessage.SendMessage(intParms.InterfaceAdministratorEmail, PII                  ", "VBECS Patient Merge Error", messageBody, "smtp.DNS   ");
				}
				catch( Exception mailexc )
				{
					string mailerrmsg = string.Concat("Unable to send mail message to Interface Administrator: ", HL7Utility.GetInnerExceptionFromException(mailexc));
                    //EventLogAppender
                    _eventsLogger.Error("VBECS Merge Interface: " + mailerrmsg);
					//System.Diagnostics.EventLog.WriteEntry( "VBECS Merge Interface", mailerrmsg, System.Diagnostics.EventLogEntryType.Error );
				}
				//
				HL7MessageLog.UpdateMessageStatus( MessageStatus.ProcessingError, this._messageControlID, exc.Message, 0, intParms.InterfaceName, Common.UpdateFunction.HL7PatientMergeInterface );
			}
			//
			return BuildAckMessage( intParms, AckCodes.AA, string.Empty );
		}

		#endregion

		#region Private Methods
		
		/// <summary>
		/// CR 2566: message is valid if merge-from OR merge-to patient is in VBECS.
		/// </summary>
		private bool ValidateVbecsPatientIdentity()
		{
			if ( _mrg == null )
				throw( new ArgumentNullException( "_mrg" ) );
			//
			if ( _pid == null )
				throw( new ArgumentNullException( "_pid" ) );
			//
			// Merge-from Patient
			// BNT 04/18/2006: Addressed CR 1820 - Changed fromPatientId from _pid[3] to _mrg[1]
			string [] fromPatientIds = GetPatientIdFromPatientIdList(HL7Utility.ParseGetStringArray( _mrg[1], _delimiters[2].ToString()) );
			//
			// CR 2566: adding ICN
			_fromPatientIcn = fromPatientIds[0];
			_fromPatientDfn = fromPatientIds[1];
			//
			if( _fromPatientDfn != null )
			{
				DataTable dtFromPatientSearch = HL7AL.CprsOmgMessage.GetHl7Patient( null, _fromPatientDfn );
				//
				if( dtFromPatientSearch.Rows.Count == 1 )
				{
					_fromPatientInVbecs = true;
					//
					_dtPatientChange.Rows[0][Common.VbecsTables.PatientChange.FromPatientGuid] = (Guid) dtFromPatientSearch.Rows[0][Common.VbecsTables.Patient.PatientGuid];
				}
			}
			// Merge-to Patient
			// BNT 04/18/2006: Addressed CR 1820 - Changed toPatientId from _mrg[1] to _pid[3].
			string [] toPatientIds = GetPatientIdFromPatientIdList(HL7Utility.ParseGetStringArray( _pid[3], _delimiters[2].ToString()) );
			//
			// CR 2566: adding ICN and SSN
			_toPatientIcn = toPatientIds[0];
			_toPatientDfn = toPatientIds[1];
			_toPatientSsn = toPatientIds[2];
			//
			if( _toPatientDfn != null )
			{
				DataTable dtToPatientSearch = HL7AL.CprsOmgMessage.GetHl7Patient( null, _toPatientDfn );
				//
				if( dtToPatientSearch.Rows.Count > 0 )
				{
					_toPatientInVbecs = true;
					//
					_dtPatientChange.Rows[0][Common.VbecsTables.PatientChange.ToPatientGuid] = (Guid) dtToPatientSearch.Rows[0][Common.VbecsTables.Patient.PatientGuid];
				}
			}
			// 
			return ( _fromPatientInVbecs || _toPatientInVbecs );
		}

		/// <summary>
		/// CR 2566: modified to handle merge-from or merge-to patient not in VBECS
		/// </summary>
		private void UpdatePatientChangeData()
		{
			_dtPatientChange.Rows[0][Common.VbecsTables.PatientChange.PatientChangeGuid] = System.Guid.NewGuid();
			_dtPatientChange.Rows[0][Common.VbecsTables.PatientChange.PatientChangeStatusCode] = "P";
			//
			// CR 2462
			_dtPatientChange.Rows[0][Common.VbecsTables.PatientChange.MergeFromVistaPatientId] = Convert.ToInt64( _fromPatientDfn );
			_dtPatientChange.Rows[0][Common.VbecsTables.PatientChange.MergeToVistaPatientId] = Convert.ToInt64( _toPatientDfn );
			//
			_dtPatientChange.Rows[0][Common.VbecsTables.PatientChange.LastUpdateUser] = LAST_UPDATE_USER_VALUE;
			//
			// Added LastUpdateFunctionId for CR 185
			_dtPatientChange.Rows[0][Common.VbecsTables.PatientChange.LastUpdateFunctionId] = Common.UpdateFunction.HL7PatientMergeInterface;
			//
			if ( !_fromPatientInVbecs ) 
			{
				// CR 2566
				// Merge-From Patient is not in VBECS so save data from the HL7: 
				// Patient ICN, DFN and Name
				//
				_dtPatientChange.Rows[0][Common.VbecsTables.PatientChange.FromPatientIcn] = _fromPatientIcn;
				//
				// CR 2931: if merge-from name contains the invalid indicator, we ignore this value
				if (_mrg[7].IndexOf(INVALID_MERGE_FROM_PATIENT_NAME_INDICATOR) == -1)
				{
					string [] patientName = HL7Utility.ParseGetStringArray( _mrg[7], _delimiters[1].ToString() );
					//
					_dtPatientChange.Rows[0][Common.VbecsTables.PatientChange.FromPatientLastName] = patientName[0];
					_dtPatientChange.Rows[0][Common.VbecsTables.PatientChange.FromPatientFirstName] = patientName[1];
					//
					if ( patientName[2].Length > 0 )
					{
						_dtPatientChange.Rows[0][Common.VbecsTables.PatientChange.FromPatientMIddleName] = patientName[2];
					}
				}	
			}
			else if ( !_toPatientInVbecs )
			{
				// CR 2566
				// Merge-To Patient is not in VBECS so save the data from the HL7: 
				// Patient ICN, DFN, SSN, DOB, Sex Code and Name
				//
				_dtPatientChange.Rows[0][Common.VbecsTables.PatientChange.ToPatientIcn] = _toPatientIcn;
				//
				_dtPatientChange.Rows[0][Common.VbecsTables.PatientChange.ToPatientSsn] = _toPatientSsn;
				//
				string[] patientName = HL7Utility.ParseGetStringArray( _pid[5], _delimiters[1].ToString() );
				//
				_dtPatientChange.Rows[0][Common.VbecsTables.PatientChange.ToPatientLastName] = patientName[0];
				_dtPatientChange.Rows[0][Common.VbecsTables.PatientChange.ToPatientFirstName] = patientName[1];
				//
				if ( patientName[2].Length > 0 )
				{
					_dtPatientChange.Rows[0][Common.VbecsTables.PatientChange.ToPatientMIddleName] = patientName[2];
				}
				//
				_dtPatientChange.Rows[0][Common.VbecsTables.PatientChange.ToPatientSexCode] = _pid[8];
				//
				_dtPatientChange.Rows[0][Common.VbecsTables.PatientChange.ToPatientDob] = HL7DateFormat.ConvertHL7DateTime(_pid[7]);
			}
			//
			HL7AL.PatientChange.InsertPatientChange(_dtPatientChange.Rows[0]);
		}

		/// <summary>
		/// CR 2566: modified to return ICN (NI) and SSN (SS) as well
		/// </summary>
		private string [] GetPatientIdFromPatientIdList(string[] patientIdList)
		{
			string [] returnValue = new string[3];
			//
			for (int i = 0; i < patientIdList.Length ; i++ )
			{
				string[] idComponents = HL7Utility.ParseGetStringArray( patientIdList[i], _delimiters[1].ToString() );
				//
				switch(idComponents[4])
				{
					case "NI":
						returnValue[0] = HL7Utility.ConvertString( idComponents[0] );
						break;
					case "PI":
						returnValue[1] = HL7Utility.ConvertString( idComponents[0] );
						break;
					case "SS":
						returnValue[2] = HL7Utility.ConvertString( idComponents[0] );
						break;
					default:
						break;
				}
			}
			return returnValue;
		}

		/// <summary>
		/// LoadDataFromHL7Message
		/// </summary>
		private void LoadDataFromHL7Message()
		{
			_delimiters = HL7Utility.ParseGetMessageDelimiters( this._message );

			string[] _hl7Segments = HL7Utility.ParseGetAllMessageSegments( this._message );
			for (int i = 0 ; i < _hl7Segments.Length - 1 ; i++ )
			{
				string[] seg = _hl7Segments[i].Split(_delimiters[0]);
				switch(seg[0])
				{
					case "MSH":
						_msh = _hl7Segments[i].Split(_delimiters[0]);
						break;

					case "PID":
						_pid = _hl7Segments[i].Split(_delimiters[0]);						
						break;

					case "MRG":
						_mrg = _hl7Segments[i].Split(_delimiters[0]);
						break;

					default:
						break;
				}
			}
		}

		/// <summary>
		/// BuildAckMessage
		/// </summary>
		private HL7AckMessage BuildAckMessage(HL7Interface intParms, AckCodes ackCode, string errorText)
		{
			StringBuilder _ack = new StringBuilder();
			_ack.Append(BuildMSHSegment(intParms));
			_ack.Append("MSA");
			_ack.Append(intParms.FieldSeparator);
			_ack.Append(ackCode.ToString());
			_ack.Append(intParms.FieldSeparator);
			_ack.Append(HL7Utility.GetMessageControlID(this._message));
			_ack.Append(intParms.FieldSeparator);
			_ack.Append(errorText);
			_ack.Append(CARRIAGE_RETURN);

			return new HL7AckMessage(_ack.ToString());
		}

		/// <summary>
		/// BuildMSHSegment
		/// </summary>
		private static string BuildMSHSegment(HL7Interface intParms)
		{
			StringBuilder _msh = new StringBuilder();
			_msh.Append("MSH");										// MSH Segment ID
			_msh.Append(intParms.FieldSeparator);					// Field Separator
			_msh.Append(intParms.EncodingCharacters);				// Encoding Characters
			_msh.Append(intParms.FieldSeparator);
			_msh.Append(intParms.VbecsApplicationId.Trim());		// Sending Application
			_msh.Append(intParms.FieldSeparator);
			_msh.Append(intParms.VbecsFacilityId.Trim());			// Sending Facility
			_msh.Append(intParms.FieldSeparator);
			_msh.Append(intParms.InterfaceApplicationId.Trim());	// Receiving Application
			_msh.Append(intParms.FieldSeparator);
			_msh.Append(intParms.InterfaceFacilityId.Trim());		// Receiving Facility
			_msh.Append(intParms.FieldSeparator);
			_msh.Append(System.DateTime.Now.ToString("yyyyMMddhhmmsszzz").Replace(":",""));		// HL7 DateTime
			_msh.Append(intParms.FieldSeparator);
			_msh.Append(intParms.FieldSeparator);
			_msh.Append(ACK_MESSAGE_TYPE);							// ACK Message Type
			_msh.Append(intParms.EncodingCharacters[0]);
			_msh.Append(ACK_TRIGGER_EVENT);							// ACK Trigger Event
			_msh.Append(intParms.FieldSeparator);
			_msh.Append("VBECS" + System.DateTime.Now.ToString("yMMddhhssffff"));		// Message Control ID
			_msh.Append(intParms.FieldSeparator);
			_msh.Append(intParms.ProcessingId);						// Processing ID
			_msh.Append(intParms.FieldSeparator);
			_msh.Append(intParms.VersionId.Trim());					// HL7 Version ID
			_msh.Append(CARRIAGE_RETURN);

			return _msh.ToString();
		}

		#endregion
	}
}
